/*
 *  							PS/2 Keyer
 *
 *  Copyright (C) 2009  David Bern, W2LNX     W2LNX@ARRL.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 *  USA, or see <http://www.gnu.org/licenses/>.
 */

#include "device.h"
#include "configuration.h"
#include "constants.h"
#include "cw_paddle_mouse.h"
#include "ps2_protocol_mouse.h"
#include "ps2_mouse.h"
#include "interrupts.h"
#include "main.h"
#include <assert.h>

#if DEBUG == 1
#define DPUTCHAR(X) 	putchar(X);
#else
#define DPUTCHAR(X)	
#endif

/******************************************************************************/
/*																			  */
/*		These routines emulate a PS/2 mouse with a CW paddle.  The left		  */
/*		and right paddles are used to generate mouse button clicks and		  */
/*		and to move the mouse pointer.										  */
/*																			  */		
/******************************************************************************/

enum {		/* routine parameters */

	TUNING_PARM = 35,
	SLEEP_PARM 	= 100,
	
	FAST_SPEED_PARM   = 30,
	SLOW_SPEED_FACTOR = 1,
	FAST_SPEED_FACTOR = 8,

	CONTACT_SIZE = 3, 
};

/* -------------------------------------------------------------------------- */
	
enum {		/* mouse device codes */

	NO_MOUSE_MOVEMENT		= 0x00,

	MOVE_MOUSE_RIGHT		= 0x08,
	MOVE_MOUSE_LEFT			= 0x18,
	MOVE_MOUSE_UP			= 0x08,
	MOVE_MOUSE_DOWN			= 0x28,

	PRESS_LEFT_BUTTON		= 0x09,
	PRESS_RIGHT_BUTTON		= 0x0A,
	PRESS_MIDDLE_BUTTON		= 0x0C,
	RELEASE_BUTTON			= 0x08
};

/* -------------------------------------------------------------------------- */

enum {	/* paddle contact modes */

	RELEASED = 'r',		/* released contact */
	CLOSED	 = 'c',		/* transition from released to pressed */
	PRESSED  = 'p',		/* holding contact */
	OPENED	 = 'o',		/* transition from pressed to released */
};

/* -------------------------------------------------------------------------- */

struct Direction {
	unsigned int command;
	signed int dx;
	signed int dy;
};

struct Direction direction[ ] =
{
	/*  0 */ { 					  MOVE_MOUSE_UP,    0,  4 },
	/*  1 */ { MOVE_MOUSE_RIGHT | MOVE_MOUSE_UP,    1,  3 },
	/*  2 */ { MOVE_MOUSE_RIGHT | MOVE_MOUSE_UP,    2,  2 },
	/*  3 */ { MOVE_MOUSE_RIGHT | MOVE_MOUSE_UP,    3,  1 }
	/*  4 */ { MOVE_MOUSE_RIGHT,				    4,  0 }, 
	/*  5 */ { MOVE_MOUSE_RIGHT | MOVE_MOUSE_DOWN,  3, -1 },
	/*  6 */ { MOVE_MOUSE_RIGHT | MOVE_MOUSE_DOWN,  2, -2 },
	/*  7 */ { MOVE_MOUSE_RIGHT | MOVE_MOUSE_DOWN,  1, -3 },
	/*  8 */ { 					  MOVE_MOUSE_DOWN,  0, -4 }, 
	/*  9 */ { MOVE_MOUSE_LEFT  | MOVE_MOUSE_DOWN, -1, -3 },
	/* 10 */ { MOVE_MOUSE_LEFT  | MOVE_MOUSE_DOWN, -2, -2 },
	/* 11 */ { MOVE_MOUSE_LEFT  | MOVE_MOUSE_DOWN, -3, -1 }, 
	/* 12 */ { MOVE_MOUSE_LEFT,					   -4,  0 }, 
	/* 13 */ { MOVE_MOUSE_LEFT  | MOVE_MOUSE_UP,   -3,  1 },
	/* 14 */ { MOVE_MOUSE_LEFT  | MOVE_MOUSE_UP,   -2,  2 },
	/* 15 */ { MOVE_MOUSE_LEFT  | MOVE_MOUSE_UP,   -1,  3 },
};

enum {	/* absolute directions */
	UP 		   = 0,
	RIGHT_UP   = 2,
	RIGHT	   = 4,
	RIGHT_DOWN = 6,
	DOWN	   = 8,
	LEFT_DOWN  = 10,
	LEFT	   = 12,
	LEFT_UP	   = 14
};

enum {
	direction_size = sizeof(direction)/sizeof(direction[0])
};

/* -------------------------------------------------------------------------- */

static int left_contact[CONTACT_SIZE];
static int right_contact[CONTACT_SIZE];

/* -------------------------------------------------------------------------- */

static int do_paddle_mouse(void);

static void shift_array(int contact[ ]);

static unsigned int reverse_direction(unsigned int paddle_direction);

static unsigned int diagonal_direction(unsigned int left_paddle_direction, 
	unsigned int right_paddle_direction);

static void move_cursor(unsigned int command, signed int dx, signed int dy);

static int is_paddle_pressed(int contact[ ]);

static int was_paddle_clicked(int contact[ ]);

static int was_paddle_closed(int contact[ ]);

static int was_paddle_opened(int contact[ ]);

static int was_paddle_released(int contact[ ]);

static void click_mouse_button(int button);

static void input_cw_paddle_mouse(int *left_contact, int *right_contact);

/* -------------------------------------------------------------------------- */

/*
 * this routine initializes the CW paddle mouse routines
 */

void initialize_cw_paddle_mouse(void)
{
	unsigned int i;

#if TESTING == 1
	printf("initialize_cw_paddle_mouse():\r\n");
#endif

	/* initialize array */
	for (i = 0; i < CONTACT_SIZE; i++) {
		Left_contact[i] = RELEASED;
	}
	for (i = 0; i < CONTACT_SIZE; i++) {
		Right_contact[i] = RELEASED;
	}

#if TESTING == 1
	printf("initialize_cw_paddle_mouse() done\r\n");
#endif
}

/* -------------------------------------------------------------------------- */

/*
 * this routine calls the routine that emulates a three button mouse
 */

int cw_paddle_mouse(void)
{
	int cw_paddle_mode;
		
	for (;;) {

		/* an interrupt wakes up processor */
		sleep();
		DPUTCHAR('#');

		/* a cw paddle was pressed */
		assert (get_interrupt_event() == PADDLE_INTERRUPT);
		cw_paddle_mode = do_paddle_mouse();
		if (cw_paddle_mode != PADDLE_MOUSE_MODE)
			break;

	}

	return cw_paddle_mode;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine emulates a three button mouse using a CW keyer paddles
 */

static int do_paddle_mouse(void)
{
	static unsigned int left_paddle_direction = LEFT;
	static unsigned int right_paddle_direction = DOWN;

	int cw_paddle_mode = PADDLE_MOUSE_MODE;
	unsigned int exit_count = 0;
	unsigned int current_direction = 0;
	unsigned int speed_factor = SLOW_SPEED_FACTOR;
	unsigned long press_count = 0;
	unsigned long sleep_count = 0;
	unsigned int command;
	unsigned int i;
	int dx;
	int dy;

	for (;;) {

		shift_array(Left_contact);
		shift_array(Right_contact);
		input_cw_paddle_mouse(&Left_contact[CONTACT_SIZE - 1], 
			&Right_contact[CONTACT_SIZE - 1]);

		/* click middle button */
		if (was_paddle_clicked(Left_contact)  == TRUE && 
			was_paddle_clicked(Right_contact) == TRUE) {

			DPUTCHAR('M');

			/*
			 *  middle mouse button is not implemented
			 */

			exit_count = 0;
			continue;

		/* click left button */
		} else if (is_paddle_pressed(Right_contact) == FALSE &&
					was_paddle_clicked(Left_contact)  == TRUE) {

			DPUTCHAR('L');

			/* exit sequence is click left, left, left paddles */
			if (exit_count++ >= 2) {

				DPUTCHAR('!');

				cw_paddle_mode = PADDLE_KEYBOARD_MODE;
				break;
			}

			click_mouse_button(PRESS_LEFT_BUTTON);
			continue;

		/* click right button */
		} else if (is_paddle_pressed(Left_contact) == FALSE && 
				   was_paddle_clicked(Right_contact) == TRUE) {
			
			DPUTCHAR('R');

			click_mouse_button(PRESS_RIGHT_BUTTON);
			continue;

		/* press left paddle: go left or right */
		} else if (is_paddle_pressed(Left_contact)  == TRUE &&	
				   was_paddle_closed(Left_contact)  == TRUE &&
				   is_paddle_pressed(Right_contact) == FALSE) {

			exit_count = 0;

			press_count = 0;
			if (speed_factor != SLOW_SPEED_FACTOR) {
				speed_factor = SLOW_SPEED_FACTOR;
			} else {
				left_paddle_direction = 
					reverse_direction(left_paddle_direction);
				current_direction = left_paddle_direction;
			}

		/* continue pressing left paddle when right paddles was released:
				continue going left or right */
		} else if (is_paddle_pressed(Left_contact)  == TRUE  &&
				   was_paddle_opened(Left_contact)  == FALSE &&
				   was_paddle_opened(Right_contact) == TRUE) {

			exit_count = 0;

			press_count = 0;
			current_direction = left_paddle_direction;

		/* continue pressing left paddle: continue going left or right */
		} else if (is_paddle_pressed(Left_contact)  == TRUE  &&
				   was_paddle_closed(Left_contact)  == FALSE &&
				   is_paddle_pressed(Right_contact) == FALSE) {
	
			exit_count = 0;

			if (press_count++ == FAST_SPEED_PARM) {
				speed_factor = FAST_SPEED_FACTOR;
			}

		/* press right paddle: go up or down */
		} else if (is_paddle_pressed(Right_contact) == TRUE &&
				   was_paddle_closed(Right_contact) == TRUE &&
				   is_paddle_pressed(Left_contact) 	== FALSE) {

			exit_count = 0;

			press_count = 0;
			if (speed_factor != SLOW_SPEED_FACTOR) {
				speed_factor = SLOW_SPEED_FACTOR;
			} else {
				right_paddle_direction = 
					reverse_direction(right_paddle_direction);
				current_direction = right_paddle_direction;
			}

		/* continue pressing right paddle when left paddle was released:
				continue going up or down */
		} else if (is_paddle_pressed(Right_contact) == TRUE  &&
				   was_paddle_opened(Right_contact) == FALSE &&
				   was_paddle_opened(Left_contact)  == TRUE) {

			exit_count = 0;

			press_count = 0;
			current_direction = right_paddle_direction;

		/* continue pressing right paddle: continue going up or down */
		} else if (is_paddle_pressed(Right_contact) == TRUE  &&
				   was_paddle_closed(Right_contact) == FALSE &&
				   is_paddle_pressed(Left_contact)  == FALSE ) {

			exit_count = 0;

			if (press_count++ == FAST_SPEED_PARM) {
				speed_factor = FAST_SPEED_FACTOR;
			}

		/* press left paddle and then right paddle: go on diagonal */
		} else if (is_paddle_pressed(Left_contact)  == TRUE  &&
				   was_paddle_closed(Left_contact)  == FALSE &&
				   is_paddle_pressed(Right_contact) == TRUE  &&
				   was_paddle_closed(Right_contact) == TRUE) {

			exit_count = 0;

			right_paddle_direction = reverse_direction(right_paddle_direction);
			current_direction = diagonal_direction(left_paddle_direction, 
				right_paddle_direction);

		/* press right paddle and then left paddle: go on diagonal */
		} else if (is_paddle_pressed(Right_contact) == TRUE  &&
				   was_paddle_closed(Right_contact) == FALSE &&
				   is_paddle_pressed(Left_contact)  == TRUE  &&
				   was_paddle_closed(Left_contact)  == TRUE) {

			exit_count = 0;

			left_paddle_direction = reverse_direction(left_paddle_direction);
			current_direction = diagonal_direction(left_paddle_direction, 
				right_paddle_direction);

		/* continue on diagonal */
		} else if (is_paddle_pressed(Left_contact)  == TRUE &&
				   is_paddle_pressed(Right_contact) == TRUE) {

			exit_count = 0;

			if (press_count++ == FAST_SPEED_PARM) {
				speed_factor = FAST_SPEED_FACTOR;
			}

		/* pause */
		} else if (was_paddle_released(Left_contact)  == TRUE &&
				   was_paddle_released(Right_contact) == TRUE) {

			DPUTCHAR('T');

			if (sleep_count++ == SLEEP_PARM)
				break;

			speed_factor = SLOW_SPEED_FACTOR;
			press_count = 0;
			continue;

		/* pause */
		} else if (is_paddle_pressed(Left_contact)  == FALSE &&
				   is_paddle_pressed(Right_contact) == FALSE) {

			DPUTCHAR('S');

			speed_factor = SLOW_SPEED_FACTOR;
			press_count = 0;
			sleep_count = 0;
			continue;

		/* unexpected condition */
		} else {

			assert (("unexpected condition", FALSE));

		}
		
		/* move the mouse cursor */
		command = direction[current_direction].command;
		dx = direction[current_direction].dx;
		dy = direction[current_direction].dy;
		for (i = 0; i < speed_factor; i++) {
			move_cursor(command, dx, dy);
		}

	} /* for */

	return cw_paddle_mode;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine shifts all the elements of the array by one
 */

static void shift_array(int contact[ ])
{
	unsigned int i;

	for (i = 0; i < CONTACT_SIZE - 1; i++) {
		contact[i] = contact[i+1];
	}
}

/* -------------------------------------------------------------------------- */

/*
 * this routine reverses the paddle direction
 */

static unsigned int reverse_direction(unsigned int paddle_direction)
{

	if (paddle_direction == LEFT) {
		paddle_direction = RIGHT;
	} else if (paddle_direction ==  RIGHT) {
		paddle_direction = LEFT;
	} else if (paddle_direction == UP) {
		paddle_direction = DOWN;
	} else if (paddle_direction == DOWN) {
		paddle_direction = UP;
	} else {
		assert (("unexpected function argument", FALSE));
	}

	return paddle_direction;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine reverses the paddle direction
 */

static unsigned int diagonal_direction(unsigned int left_paddle_direction, 
	unsigned int right_paddle_direction)
{
	unsigned int current_direction;

	if (left_paddle_direction == RIGHT) {
		if (right_paddle_direction == UP) {
			current_direction = RIGHT_UP;
		} else if (right_paddle_direction == DOWN) {
			current_direction = RIGHT_DOWN;
		} else {
			assert (("unexpected function argument", FALSE));
		}
	} else if (left_paddle_direction == LEFT){
		if (right_paddle_direction == UP) {
			current_direction = LEFT_UP;
		} else if (right_paddle_direction == DOWN) {
			current_direction = LEFT_DOWN;
		} else {
			assert (("unexpected function argument", FALSE));
		}
	} else {
		assert (("unexpected function argument", FALSE));
	}

	return current_direction;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine moves the mouse cursor
 */

static void move_cursor(unsigned int command, signed int dx, signed int dy)
{

	send_to_host_mouse(command);
	send_to_host_mouse(dx);
	send_to_host_mouse(dy);
}

/* -------------------------------------------------------------------------- */

/*
 * this function returns true if a paddle is pressed and held closed
 */

static int is_paddle_pressed(int contact[ ])
{
	int pressed_flag = FALSE;

	if (contact[CONTACT_SIZE - 1] == PRESSED &&
		contact[CONTACT_SIZE - 2] == PRESSED) {
		pressed_flag =  TRUE;
	}

	return pressed_flag;
}

/* -------------------------------------------------------------------------- */

/*
 * this function returns true if a paddle was clicked, i. e. momentarily closed
 */

static int was_paddle_clicked(int contact[ ])
{
	int clicked_flag = FALSE;

	if (contact[1] == CLOSED && contact[CONTACT_SIZE - 1] == OPENED) {
		clicked_flag = TRUE;
	}

	return clicked_flag;
}

/* -------------------------------------------------------------------------- */

/*
 * this function returns true if a paddle was just closed
 */

static int was_paddle_closed(int contact[ ])
{
	int closed_flag = FALSE;

	if (contact[0] == CLOSED) {
		closed_flag = TRUE;
	}

	return closed_flag;
}

/* -------------------------------------------------------------------------- */

/*
 * this function returns true if a paddle was just opened
 */

static int was_paddle_opened(int contact[ ])
{

	int opened_flag = FALSE;

	if (contact[CONTACT_SIZE - 1] == OPENED) {
		opened_flag = TRUE;
	}

	return opened_flag;
}

/* -------------------------------------------------------------------------- */

/*
 * this function returns true if a paddle was released 
 */

static int was_paddle_released(int contact[ ])
{
	int released_flag = FALSE;

	if (contact[0] == RELEASED) {
		released_flag = TRUE;
	} 

	return released_flag;
}

/* -------------------------------------------------------------------------- */
/*
 * this routine clicks a button on the mouse
 */

static void click_mouse_button(int button)
{

	send_to_host_mouse(button);
	send_to_host_mouse(NO_MOUSE_MOVEMENT);
	send_to_host_mouse(NO_MOUSE_MOVEMENT);
	delay_ms(50);
	send_to_host_mouse(RELEASE_BUTTON);
	send_to_host_mouse(NO_MOUSE_MOVEMENT);
	send_to_host_mouse(NO_MOUSE_MOVEMENT);
}

/* -------------------------------------------------------------------------- */

/*
 * this routine returns the value of the left and right contacts of the paddle
 */

static void input_cw_paddle_mouse(int *left_contact, int *right_contact)
{
	unsigned int port_B;
	int prev_left_contact;
	int prev_right_contact;
	int left_contact_flag = FALSE;
	int right_contact_flag = FALSE;

	prev_left_contact = *left_contact;
	prev_right_contact = *right_contact;
	*left_contact = RELEASED;
	*right_contact = RELEASED;

	if (get_interrupt_event() == PADDLE_INTERRUPT) { 
		DPUTCHAR('*');

		clear_interrupt_event();

		port_B = get_port_B();
		delay_ms(DEBOUNCE_DELAY);
	} else {
		port_B = input_B();
		delay_ms(TUNING_PARM);
	}

	if (bit_test(Port_B, paddle_left_contact_bit) == LOW) {
		left_contact_flag = TRUE;
	} 
	if (bit_test(Port_B, paddle_right_contact_bit) == LOW) {
		right_contact_flag = TRUE;
	}
	output_bit(STATUS_LED, left_contact_flag || right_contact_flag);

	if (left_contact_flag == TRUE) {
		DPUTCHAR('<');
		switch (prev_left_contact) {
		case OPENED:
		case RELEASED:	
			*left_contact = CLOSED;
			break;
		case CLOSED:
		case PRESSED:
			*left_contact = PRESSED;
			break;
		default:
			break;
		}

	} else {
		switch (prev_left_contact) {
		case CLOSED:
		case PRESSED:
			*left_contact = OPENED;
			break;
		case OPENED:
		case RELEASED:
			*left_contact = RELEASED;
			break;
		default:
			break;
		}
	}

	if (right_contact_flag == TRUE) {
		DPUTCHAR('>');
		switch (prev_right_contact) {
		case OPENED:
		case RELEASED:
			*right_contact = CLOSED;
			break;
		case CLOSED:
		case PRESSED:
			*right_contact = PRESSED;
			break;
		default:
			break;
		}

	} else {
		switch (prev_right_contact) {
		case CLOSED:
		case PRESSED:
			*right_contact = OPENED;
			break;
		case OPENED:
		case RELEASED:	
			*right_contact = RELEASED;
			break;
		default:
			break;
		}
	}
}

/* -------------------------------------------------------------------------- */

/*
 * this routine demonstrates the PS/2 mouse by moving the mouse cursor 
 * in opposing circles
 */

void demo_ps2_mouse(void)
{
	signed int i;
	unsigned int j;
	unsigned int command;
	signed int dx;
	signed int dy;

	printf("demonstrate PS/2 mouse emulation\r\n");

	for (i = 0; i < direction_size; i++) {
		/* move the mouse cursor */
		command = direction[i].command;
		dx = direction[i].dx;
		dy = direction[i].dy;
		for (j = 0; j < 2*FAST_SPEED_FACTOR; j++) {
			move_cursor(command, dx, dy);
			delay_ms(1);
		}
	}

	for (i = direction_size - 1;  i >= 0; i--) {
		/* move the mouse cursor */
		command = direction[i].command;
		dx = direction[i].dx;
		dy = direction[i].dy;
		for (j = 0; j < 2*FAST_SPEED_FACTOR; j++) {
			move_cursor(command, dx, dy);
			delay_ms(1);
		}
	}
}

/* -------------------------------------------------------------------------- */

#if TESTING == 1
/* ========================================================================== */
/*																			  */
/*								TEST ROUTINES								  */
/*																			  */			
/* ========================================================================== */

/*
 * this test routine tests the emulated mouse movement using the keyer paddles
 */


void test_cw_paddle_mouse(void)
{

	blink_status_led(2);
	printf("click on the Windows desktop\r\n");
	delay_ms(1000);

	cw_paddle_mouse();	/* ignore return value */
}

/* -------------------------------------------------------------------------- */

#endif
